/* --------------------------------------------------------------
  main.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


import Router from './libs/Router'; 
import Debug from './libs/Debug'; 
import Validator from './libs/Validator'; 
import UrlParser from './libs/UrlParser'; 
import LanguageApi from './api/LanguageApi'; 
import ConfigApi from './api/ConfigApi';
import Environment from './libs/Environment';
import Config from './libs/Config'; 
import Language from './libs/Language'; 
import TemplatePreview from './widgets/TemplatePreview'; 
import PreviewType from './entities/PreviewType'; 
import PreviewResizer from './widgets/PreviewResizer'; 
import AppFrame from './widgets/AppFrame'; 
import FrameToggle from './widgets/FrameToggle';
import 'babel-polyfill';
import enableAboutInformation from './libs/About';

/**
 * StyleEdit3 - Main App File
 *
 * This is the JS entry point for the StyleEdit3 project. It will start the required controllers
 * and widgets in order to setup the UI. There are various controllers that serve different
 * views & layers of StyleEdit. The routing is done with the "Router" module which will load the
 * required controller upon request.
 *
 * This file will create a new window object with the name "StyleEdit". Use this as the namespace
 * object for the whole application in order to avoid the global namespace pollution.
 *
 * Namely after initialization there will be the following instances available at any time:
 *
 *   - window.StyleEdit.Config >> Provides configuration information.
 *   - window.StyleEdit.Language >> Contains the required translations.
 *   - window.StyleEdit.Debug >> Use this when debugging the application.
 *   - window.StyleEdit.Validator >> Use this module for validating argument values
 *
 * Add the "debug" GET parameter to enable the debug environment of the JavaScript code. In the
 * debug environment the unminified files are loaded along with source maps so that you can see
 * and debug through the original files.
 *
 * Notice: The JS section of StyleEdit must not require any configuration value from the
 * PHP files. All the configuration must be done through GET parameters. This makes the app
 * more flexible between different projects and templates.
 *
 * Configuration GET parameters:
 *
 *   - template >> The template name that is configured (required).
 *   - lang >> Defines the selected interface language (must be an ISO code like "de" or "en") - (optional).
 *   - debug >> Will toggle the debug environment of JavaScript code (optional).
 *
 * During the lifecycle of the application some specific events are triggered so that various objects
 * can react accordingly. It is better to catch these event at the body or the document object in order
 * to be sure that they will be reachable from any place.
 *
 *   - StyleEdit.StyleCreated >> Triggered after a style is created.
 *   - StyleEdit.StyleUpdated >> Triggered after a style is saved.
 *   - StyleEdit.StyleDeleted >> Triggered after a style is removed.
 *   - StyleEdit.StyleActivated >> Triggered after a style has been activated.
 *   - StyleEdit.CustomStylesChanged >> Triggered after custom styles of a style were changed.
 *   - StyleEdit.ControllerInitialized >> Triggered after a successful controller initialization.
 *   - StyleEdit.ControllerDestroyed >> Triggered after a successful controller destruction.
 *   - StyleEdit.Terminated >> Triggered after StyleEdit is terminated and before the browser redirection to the
 *     preview page.
 *   - StyleEdit.PreviewReloaded >> Triggered after the template preview has finished reloading.
 */
$(function() {
	
	'use strict';
	
	// ------------------------------------------------------------------------
	// SETUP ENVIRONMENT
	// ------------------------------------------------------------------------
	
	Environment.setup();
	
	// ------------------------------------------------------------------------
	// INITIALIZE APPLICATION
	// ------------------------------------------------------------------------
	
	// Setup the required configuration. 
	const config = {
		version: '3.0',
		baseUrl: (window.location.origin + window.location.pathname).replace(/\/StyleEdit3\/.*$/, '/StyleEdit3'),
		previewUrl: window.location.origin + window.location.pathname.replace(/\/StyleEdit3\/.*$/, ''),
		environment: UrlParser.getParam('debug') !== undefined ? 'development' : 'production',
		languageCode: UrlParser.getParam('lang') || 'de',
		theme: UrlParser.getParam('theme') !== undefined,
		template: UrlParser.getParam('template'),
		tempStyleName: '__temporary_style',
		localStorageColorPalette: 'style_edit_color_palette'
	};
	
	if (!config.template) {
		throw new Error('Template GET parameter is missing, please append your request URL with the '
			+ '"&template=TemplateName".');
	}
	
	// Create the StyleEdit namespace object. 
	window.StyleEdit = {
		Config: new Config(config),
		Language: new Language(),
		Debug: Debug,
		Validator: Validator
	};
	
	// Add the "About" property in the StyleEdit object (only in development environment).
	enableAboutInformation(StyleEdit);
	
	// Fetch initial data from the server.  
	const requests = [
		ConfigApi.get(),
		LanguageApi.getTranslations(StyleEdit.Config.get('languageCode'))
	]; 
	
	// Initialize the main UI elements when AJAX requests are done. 
	$.when.apply($, requests)
		.done(function() {
			const finalConfig = $.extend(true, {}, config, arguments[0][0]); 
			StyleEdit.Config = new Config(finalConfig); // Reinitialize with final config values. 
			StyleEdit.Language.addSection('style_edit', arguments[1][0]); // Add translations section.
			
			// Create a TemplatePreview Instance
			const templatePreview = new TemplatePreview($('.style-edit-preview'), StyleEdit.Config.get('previewUrl'));
			
			// Create a PreviewResizer Instance
			const previewResizer = new PreviewResizer($('.preview-resizer'), templatePreview);
			const desktopPreview = new PreviewType();
			const tabletPreviewLandscape = new PreviewType();
			const tabletPreviewPortrait = new PreviewType();
			const mobilePreviewPortrait = new PreviewType();
			
			desktopPreview
				.setClass('desktop')
				.setName(StyleEdit.Language.translate('option_desktop_preview', 'style_edit'))
				.setPreviewCss({
					width: '100%',
					height: '100%',
					padding: '0',
					position: '',
					top: '',
					left: '',
					margin: '',
					background: ''
				})
				.setIconClass('fa fa-desktop')
				.setResolution('> 1200');
			
			tabletPreviewLandscape
				.setClass('tablet landscape')
				.setName(StyleEdit.Language.translate('option_tablet_preview_landscape', 'style_edit'))
				.setPreviewCss({
					width: '1024px',
					height: '768px',
					padding: '20px 78px 21px 25px',
					position: 'absolute',
					top: '50%',
					left: '50%',
					margin: '-405px 0 0 -723px',
					background: 'url(' + StyleEdit.Config.get('baseUrl')
					+ '/assets/images/tablet-landscape-preview-frame.svg) 0% 0% / contain no-repeat'
				})
				.setIconClass('fa fa-tablet fa-rotate-270')
				.setResolution('1024x768');
			
			tabletPreviewPortrait
				.setClass('tablet portrait')
				.setName(StyleEdit.Language.translate('option_tablet_preview_portrait', 'style_edit'))
				.setPreviewCss({
					width: '768px',
					height: '1024px',
					padding: '24px 25px 75px',
					position: 'absolute',
					top: '50%',
					left: '50%',
					margin: '-455px 0 0 -544px',
					background: 'url(' + StyleEdit.Config.get('baseUrl')
					+ '/assets/images/tablet-portrait-preview-frame.svg) 50% 0% / contain no-repeat'
				})
				.setIconClass('fa fa-tablet')
				.setResolution('768x1024');
			
			mobilePreviewPortrait
				.setClass('mobile portrait')
				.setName(StyleEdit.Language.translate('option_mobile_preview_portrait', 'style_edit'))
				.setPreviewCss({
					width: '375px',
					height: '627px',
					padding: '54px 25px 75px',
					position: 'absolute',
					top: '50%',
					left: '50%',
					margin: '-378px 0 0 -372px',
					background: 'url(' + StyleEdit.Config.get('baseUrl')
					+ '/assets/images/mobile-portrait-preview-frame.svg) 50% 0% / contain no-repeat'
				})
				.setIconClass('fa fa-mobile')
				.setResolution('375x627');
			
			previewResizer
				.addPreviewType(mobilePreviewPortrait)
				.addPreviewType(tabletPreviewPortrait)
				.addPreviewType(tabletPreviewLandscape)
				.addPreviewType(desktopPreview);
			
			// Create an AppFrame & FrameToggle Instance
			const frameToggle = new FrameToggle($('.style-edit-frame'), $('.frame-toggle'));
			const appFrame = new AppFrame($('.style-edit-frame'), previewResizer, frameToggle);
			appFrame.initialize();
			
			// Load the MyStylesView controller. 
			Router.load('MyStylesView');
		});
});
